/******************************************************************************
DtCtrl

Base class for DT GUI controls

Implements the following
* Timer limited Redraw rate
* Set font handler
* Mouse enter & leave
* Mouse drag

History
	Date       Version    Programmer         Comments
	29/07/03   1.0        Darrell Tam
******************************************************************************/

#if !defined(AFX_DTCTRL_H__2570BACB_3BD7_4E2C_825C_63459E42BD76__INCLUDED_)
#define AFX_DTCTRL_H__2570BACB_3BD7_4E2C_825C_63459E42BD76__INCLUDED_
#include "misc_stuff.h"

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000
// DtCtrl.h : header file
//



/////////////////////////////////////////////////////////////////////////////
// DtCtrl window

class DtCtrl : public CWnd
{
public:
	// notify codes of messages sent to parent window
	enum {
		NC_REDRAW,		// control is being changed or needs repainting in parent_draw mode
		NC_BUTTONUP,	// user has finished moving the control
		NC_INTERCEPT_MSG, // can intercept all windows messages
		NC_LAST			// last notify code
	} NOTIFY_CODES;

// Construction
public:
	DtCtrl();
	virtual ~DtCtrl();

// Operations
public:
	// set our control Id
	void setIDC(int _idc) { idc = _idc; }

	// set true if the parent window should call redrawControl_
	void parentDraw(bool flag=true) { parent_draw = flag; }

	// set the minimum time between redraws (so as to lessen CPU load)
	void setMaxRedrawPeriod(int msec) { max_redraw_period = msec; }

	// return the client rectangle
	CRect getClientRect(void) const { return client_rect; }

	// the parent can be notified of any windows message 
	void parentInterceptMsgs(bool flag=true) { parent_intercept_msgs = flag; }

	// parent can prevent message from being processed
	void parentConsumeMsg(void) { parent_consume_msg=true; }

	// access to the most current windows message for parent intercept
	int currWpMessage() const { return wp_message; }
	WPARAM currWpWparam() const { return wp_wParam; }
	LPARAM currWpLparam() const { return wp_lParam; }

public: // helper methods for derived classes 

	// send a command message to parent window
	void sendCmdMsg(unsigned short notify_code) {
		if(idc != 0xffff)
			GetParent()->SendMessage(WM_COMMAND, idc+((unsigned long)notify_code<<16), (long)m_hWnd);
	}

	// input focus control
	virtual void captureInput(bool force = false);
	virtual void releaseInput(bool force = false);

	// redraw control if need be, limited redraws per second unless force
	virtual void redrawControl(bool _force_redraw = false);

public:
	// internal call to perform actual redraw (call from parent during NC_REDRAW
	// call if parent draw is enabled)
	virtual void redrawControl_(void);

	virtual void prepareDraw(CBitmap& bitmap, CDC& dc);
	virtual void endDraw(void);

protected: // derived controls should override the following

	// do the actual drawing (into mem_bitmap)
	virtual void drawControl(void) = 0;

	// test whether drawControl() really needs to be called
	virtual bool chkRedraw(void) = 0;

	// various mouse events
	virtual void OnMouseLeave(void) = 0;
	virtual void OnLButtonDrag(void) = 0;
	virtual void OnRButtonDrag(void) = 0;
	virtual void OnMouseMove(void) = 0;
	virtual void OnLButtonDblClk(void) = 0;
	virtual void OnLButtonDown(void) = 0;
	virtual void OnLButtonUp(void) = 0;
	virtual void OnRButtonDown(void) = 0;
	virtual void OnRButtonDblClk(void) = 0;
	virtual void OnRButtonUp(void) = 0;

protected:
	// test whether the mouse has left our window
	virtual bool chkPoint(void);

	// variable initialization
	virtual void init();

	int idc;				// control id for messages sent to parent

	bool input_capture;		// mouse and keyboard captured
	bool desensitize_mouse;	// ignore small mouse movement
	bool lbutton_down;		// left button is currently pressed
	bool rbutton_down;		// right button is currently pressed
	bool parent_draw;		// parent must draw

	CRect client_rect;		// cached GetClientRect()
	CPoint curr_point;		// mouse point of most recent message
	CRect desensitize_rect;

	TChk<CFont*> font;		// font to use

	int max_redraw_period;	// maximum redraw time in msec
	bool delay_draw;		// draw pending
	bool draw_timer_set;	// waiting for draw timer to go off

	// temporaries during redraw
	bool force_redraw;		// force redraw
	CBitmap* old_bitmap;	// old bitmap
	CBitmap* mem_bitmap;	// in memory bitmap
	CDC* mem_dc;			// 

	// messages routed via parent (parent must call consumeThisMsg() to halt message)
	bool parent_intercept_msgs; 

	// temporary used during message processing
	bool parent_consume_msg; // parent can stop a message, default is to allow message through
	UINT wp_message;
	WPARAM wp_wParam;
	LPARAM wp_lParam;

// Overrides
	// ClassWizard generated virtual function overrides
	//{{AFX_VIRTUAL(DtCtrl)
	protected:
	virtual LRESULT WindowProc(UINT message, WPARAM wParam, LPARAM lParam);
	//}}AFX_VIRTUAL

public:

	// Generated message map functions
protected:

	//{{AFX_MSG(DtCtrl)
	afx_msg int OnCreate(LPCREATESTRUCT lpCreateStruct);
	afx_msg void OnLButtonDblClk(UINT nFlags, CPoint point);
	afx_msg void OnLButtonDown(UINT nFlags, CPoint point);
	afx_msg void OnLButtonUp(UINT nFlags, CPoint point);
	afx_msg void OnMouseMove(UINT nFlags, CPoint point);
	afx_msg void OnPaint();
	afx_msg void OnRButtonDown(UINT nFlags, CPoint point);
	afx_msg void OnRButtonUp(UINT nFlags, CPoint point);
	afx_msg void OnRButtonDblClk(UINT nFlags, CPoint point);
	afx_msg void OnSize(UINT nType, int cx, int cy);
	afx_msg void OnTimer(UINT nIDEvent);
	//}}AFX_MSG
	afx_msg LRESULT HandleSetFont(WPARAM wParam, LPARAM lParam);
	DECLARE_MESSAGE_MAP()
};

/////////////////////////////////////////////////////////////////////////////

//{{AFX_INSERT_LOCATION}}
// Microsoft Visual C++ will insert additional declarations immediately before the previous line.

#endif // !defined(AFX_DTCTRL_H__2570BACB_3BD7_4E2C_825C_63459E42BD76__INCLUDED_)
